/* $Id: gap.c,v 1.11 1999/02/17 22:56:35 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test gap field in trailer.

   We once had a bug with overlap in zoom mode, where the trailer gap
   field was set incorrectly.  Another we once had was with changing
   the enable settings while a measurement is running.  This test
   verifies that this is fixed, along with a bunch of other things. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024

#define	NORMAL_SPAN	10000
#define	ZOOM_SPAN_V	2000	/* For E1432 */
#define	ZOOM_SPAN_S	10000	/* For E1433 */
#define	CLOCK_FREQ	51200

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: gap.c,v 1.11 1999/02/17 22:56:35 ericb Exp $";
static const char *progname;

static void
my_usleep(unsigned int usec)
{
    struct timeval timeout;

    timeout.tv_sec = usec / 1000000;
    timeout.tv_usec = usec % 1000000;
    (void) select(0, 0, 0, 0, &timeout);
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list, int *vibrato)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    /* Do we have E1432 or E1433? */
    *vibrato = 0;
    for (i = 0; i < nmod; i++)
	if (hwconfig[i].model_code == E1432_MODEL_CODE_E1432)
	    *vibrato = 1;

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      long blocksize, long overlap, long avg,
      int data_mode, int data_size, int fft, int magsq, int zoom,
      int oversample, int enable, int delay, int vibrato, int verbose)
{
    double  span;

    if (zoom)
    {
	if (vibrato)
	    span = ZOOM_SPAN_V;
	else
	    span = ZOOM_SPAN_S;
    }
    else
	span = NORMAL_SPAN;

    if (data_size == E1432_DATA_SIZE_16)
    {
	blocksize &= ~0x1;
	overlap &= ~0x1;
    }

    if (verbose)
	(void) printf("Bs %ld olap %4ld %s size %s "
		      "fft %d magsq %d zoom %d osamp %d en %d d %d avg %2d\n",
		      blocksize, overlap,
		      data_mode == E1432_CONTINUOUS_MODE ?
		      "Cont " :
		      data_mode == E1432_DATA_MODE_OVERLAP_BLOCK ?
		      "OverB" :
		      data_mode == E1432_DATA_MODE_OVERLAP_FREERUN ?
		      "OverF" : "Block",
		      data_size == E1432_DATA_SIZE_16 ? "16 " :
		      data_size == E1432_DATA_SIZE_32 ? "32 " :
		      data_size == E1432_DATA_SIZE_32_SERV ?
		      "32S" : "32F",
		      fft, magsq, zoom, oversample, enable, delay, avg);

    CHECK(e1432_reset_measure(hw, group));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    /* Do data size before blocksize or overlap, to make sure
       blocksize or overlap don't get rounded wrong before we change
       the data size. */
    CHECK(e1432_set_data_size(hw, group, data_size));
    CHECK(e1432_set_blocksize(hw, group, blocksize));
    CHECK(e1432_set_clock_freq(hw, group, CLOCK_FREQ));
    CHECK(e1432_set_data_mode(hw, group, data_mode));
    CHECK(e1432_set_decimation_oversample(hw, group, oversample ?
					  E1432_DECIMATION_OVERSAMPLE_ON :
					  E1432_DECIMATION_OVERSAMPLE_OFF));
    CHECK(e1432_set_overlap(hw, group, overlap));
    CHECK(e1432_set_span(hw, group, span));
    CHECK(e1432_set_zoom(hw, group, zoom ? E1432_ZOOM_ON : E1432_ZOOM_OFF));

    if (fft)
    {
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
	if (magsq)
	{
	    CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_RMS));
	    CHECK(e1432_set_avg_number(hw, group, avg > 0 ? avg : 1));
	    CHECK(e1432_set_avg_update(hw, group, avg > 0 ? avg : 1));
	}
	else if (avg > 0)
	{
	    CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_LIN));
	    CHECK(e1432_set_avg_number(hw, group, avg));
	    CHECK(e1432_set_avg_update(hw, group, avg));
	}
	else
	    CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_NONE));
    }
    else
    {
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_TIME));
	CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_NONE));
    }

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type, int zoom)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }
    if (zoom)
	df2++;

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 1)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    long blocksize, long overlap, long avg,
    int data_mode, int data_size, int fft, int magsq, int zoom,
    int oversample, int enable, int delay, int vibrato, int verbose)
{
    FLOATSIZ64 *buffer;
    LONGSIZ32 count, expect, scale;
    struct e1432_trailer trailer;
    double  span;
    int     scan, type, chan, compare_equal;

    if (zoom)
    {
	if (vibrato)
	    span = ZOOM_SPAN_V;
	else
	    span = ZOOM_SPAN_S;
    }
    else
	span = NORMAL_SPAN;

    if (data_size == E1432_DATA_SIZE_16)
    {
	blocksize &= ~0x1;
	overlap &= ~0x1;
    }

    expect = blocksize;
    if (zoom)
	expect *= 2;
    buffer = malloc(expect * sizeof *buffer);
    if (buffer == NULL)
    {
	(void) fprintf(stderr, "%s: malloc(%ld) failed\n",
		       progname, blocksize * sizeof *buffer);
	return -1;
    }

    CHECK(e1432_init_measure(hw, group));

    for (scan = 0; scan < 4; scan++)
    {
	/* Wait for block available */
	CHECK(wait_block_avail(hw, group, scan, verbose,
			       avg > 0 ? blocksize * avg : blocksize,
			       span));

	/* Read the data */
	for (type = 0; type < (fft ? 2 : 1); type++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (verbose > 2)
		    (void) printf("Reading chan %d type %d\n",
				  chan, type);

		expect = blocksize;
		if (type != 0 && magsq)
		    expect /= 2;
		if (zoom)
		    expect *= 2;
		CHECK(e1432_read_float64_data(hw, chan_list[chan],
					      type == 0 ?
					      E1432_TIME_DATA :
					      E1432_FREQ_DATA,
					      buffer, expect,
					      &trailer,
					      &count));
		if (count != expect)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: "
				   "actual count was %ld\n",
				   progname, count);
		    return -1;
		}

		CHECK(check_trailer(&trailer, CLOCK_FREQ, span,
				    chan_list[chan], type, zoom));

		if (data_mode == E1432_BLOCK_MODE)
		    expect = blocksize;
		else
		    expect = blocksize - overlap;
		if (zoom)
		    expect *= 2;
		scale = 1;
		if (fft && avg > 0)
		    scale = avg;
		if (scan == 0)
		    scale--;
		expect *= scale;

		compare_equal =
		    data_mode == E1432_CONTINUOUS_MODE && scan > 0;

		if (compare_equal && trailer.gap != expect)
		{
		    (void) fprintf(stderr,
				   "%s: chan %d gap %ld != expect %ld\n",
				   progname, chan, trailer.gap, expect);
		    return -1;
		}
		if (!compare_equal && trailer.gap < expect)
		{
		    (void) fprintf(stderr,
				   "%s: chan %d gap %ld < expect %ld\n",
				   progname, chan, trailer.gap, expect);
		    return -1;
		}
	    }

	if (enable)
	{
	    /* Toggling enable might affect the gap */
	    CHECK(e1432_set_enable(hw, chan_list[0],
				   E1432_ENABLE_TYPE_TIME,
				   E1432_ENABLE_OFF));
	    if (delay)
		my_usleep(20000);
	    CHECK(e1432_set_enable(hw, chan_list[0],
				   E1432_ENABLE_TYPE_TIME,
				   E1432_ENABLE_ON));
	}
    }

    free(buffer);

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    long    blocksize, overlap, avg;
    int     dmode, data_mode, dsize, data_size, fft, magsq, zoom;
    int     enable, delay, oversample, verbose;
    int     c, i, nmod, group, nchan, loops, vibrato;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:n:N:uvV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    if (init(nmod, laddr, &hw, &group, &nchan, chan_list, &vibrato) < 0)
	return EXIT_FAILURE;

    /* Default parameter setups */
    blocksize = 1024;
    overlap = 800;
    avg = 0;
    data_mode = E1432_CONTINUOUS_MODE;
    data_size = E1432_DATA_SIZE_16;
    fft = 1;
    magsq = 0;
    zoom = 0;
    oversample = 0;
    enable = 0;
    delay = 0;

    /* This particular setup has caused problems with gap in the past */
    overlap = 0;
    for (avg = 5; avg <= 10; avg += 5)
	for (magsq = 0; magsq < 2; magsq++)
	{
	    if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
		      data_mode, data_size, fft, magsq, zoom, oversample,
		      enable, delay, vibrato, verbose) < 0)
		return EXIT_FAILURE;
	    for (loops = 0; loops < 10; loops++)
		if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
			data_mode, data_size, fft, magsq, zoom, oversample,
			enable, delay, vibrato, verbose) < 0)
		    return EXIT_FAILURE;
	}
    overlap = 800;
    avg = 0;
    magsq = 0;

    for (delay = 0; delay < 2; delay++)
	for (enable = 0; enable < 2; enable++)
	    for (dmode = 0; dmode < 4; dmode++)
	    {
		if (delay && !enable)
		    continue;

		switch (dmode)
		{
		case 0:
		    data_mode = E1432_CONTINUOUS_MODE;
		    break;
		case 1:
		    data_mode = E1432_BLOCK_MODE;
		    break;
		case 2:
		    data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
		    break;
		case 3:
		    data_mode = E1432_DATA_MODE_OVERLAP_FREERUN;
		    break;
		}

		if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
			  data_mode, data_size, fft, magsq, zoom, oversample,
			  enable, delay, vibrato, verbose) < 0)
		    return EXIT_FAILURE;
		if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
			data_mode, data_size, fft, magsq, zoom, oversample,
			enable, delay, vibrato, verbose) < 0)
		    return EXIT_FAILURE;
	    }
    data_mode = E1432_CONTINUOUS_MODE;
    enable = 0;
    delay = 0;

#if SPAN * 256 / 100 > CLOCK_FREQ - 1
    for (oversample = 0; oversample < 2; oversample++)
    {
	if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
		  data_mode, data_size, fft, magsq, zoom, oversample,
		  enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
	if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
		data_mode, data_size, fft, magsq, zoom, oversample,
		enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
    }
    oversample = 0;
#endif

#if CLOCK_FREQ < 165000
    for (dsize = 0; dsize < 4; dsize++)
    {
	switch (dsize)
	{
	case 0:
	    data_size = E1432_DATA_SIZE_16;
	    break;
	case 1:
	    data_size = E1432_DATA_SIZE_32;
	    break;
	case 2:
	    data_size = E1432_DATA_SIZE_32_SERV;
	    break;
	case 3:
	    data_size = E1432_DATA_SIZE_FLOAT32;
	    break;
	}

	if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
		  data_mode, data_size, fft, magsq, zoom, oversample,
		  enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
	if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
		data_mode, data_size, fft, magsq, zoom, oversample,
		enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
    }
    data_size = E1432_DATA_SIZE_16;
#endif

    blocksize = 2048;
    for (overlap = -401; overlap <= 401; overlap += 401)
    {
	if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
		  data_mode, data_size, fft, magsq, zoom, oversample,
		  enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
	for (loops = 0; loops < 4; loops++)
	    if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
		    data_mode, data_size, fft, magsq, zoom, oversample,
		    enable, delay, vibrato, verbose) < 0)
		return EXIT_FAILURE;
    }
    blocksize = 1024;
    overlap = 800;

#if CLOCK_FREQ < 165000
    fft = 0;
    data_size = E1432_DATA_SIZE_32;
    for (blocksize = 1021; blocksize <= 1027; blocksize += 6)
    {
	if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
		  data_mode, data_size, fft, magsq, zoom, oversample,
		  enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
	if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
		data_mode, data_size, fft, magsq, zoom, oversample,
		enable, delay, vibrato, verbose) < 0)
	    return EXIT_FAILURE;
    }
    blocksize = 1024;
    fft = 1;
    data_size = E1432_DATA_SIZE_16;
#endif

    for (avg = 0; avg <= 10; avg += 5)
	for (fft = 0; fft < 2; fft++)
	    for (magsq = 0; magsq < 2; magsq++)
		for (zoom = 0; zoom < 2; zoom++)
		{
		    if (!fft && avg > 0)
			continue;
		    if (!fft && magsq)
			continue;

		    if (setup(hw, group, nchan, chan_list, blocksize, overlap, avg,
			      data_mode, data_size, fft, magsq, zoom, oversample,
			      enable, delay, vibrato, verbose) < 0)
			return EXIT_FAILURE;
		    if (run(hw, group, nchan, chan_list, blocksize, overlap, avg,
			    data_mode, data_size, fft, magsq, zoom, oversample,
			    enable, delay, vibrato, verbose) < 0)
			return EXIT_FAILURE;
		}

    return EXIT_SUCCESS;
}
